"""Repository for Employee entities (TinyDB-backed)."""

from __future__ import annotations

import os
from datetime import date
from typing import Any, Dict, List, Optional

from tinydb import TinyDB, Query


def _iso(val: Any) -> Any:
    """Serialize date objects to ISO strings; pass through other types."""
    if isinstance(val, date):
        return val.isoformat()
    return val


class EmployeeRepository:
    """
    Minimal TinyDB repository for employees.
    Stores data in data/employees.json, table 'employees'.
    Ensures unique emp_code and serializes date_joined to ISO string.
    """

    def __init__(self, db_path: str = "data/employees.json") -> None:
        os.makedirs(os.path.dirname(db_path), exist_ok=True)
        self.db = TinyDB(db_path)
        self.table = self.db.table("employees")

    # ---------- helpers ----------
    def _next_id(self) -> int:
        rows = self.table.all()
        if not rows:
            return 1
        return max((int(r.get("id") or 0) for r in rows), default=0) + 1

    def _ensure_unique_emp_code(self, emp_code: str, exclude_id: Optional[int] = None) -> None:
        q = Query()
        existing = self.table.get(q.emp_code == emp_code)
        if existing and (exclude_id is None or existing.get("id") != exclude_id):
            raise ValueError(f"Employee code '{emp_code}' already exists.")

    # ---------- CRUD ----------
    def create_employee(self, employee: Any) -> int:
        """
        Insert an employee. Accepts dict or Pydantic model with .dict().
        Serializes date_joined to ISO string.
        """
        data: Dict[str, Any] = employee.dict() if hasattr(employee, "dict") else dict(employee)

        if not data.get("emp_code"):
            raise ValueError("emp_code is required.")

        # Uniqueness check
        self._ensure_unique_emp_code(data["emp_code"])

        # Serialize date fields
        if "date_joined" in data:
            data["date_joined"] = _iso(data["date_joined"])

        if not data.get("id"):
            data["id"] = self._next_id()

        self.table.insert(data)
        return int(data["id"])

    def update_employee(self, emp_id: int, **fields: Any) -> bool:
        """Update fields for an employee (serialize dates and enforce unique emp_code)."""
        # Enforce unique emp_code if being updated
        if "emp_code" in fields and fields["emp_code"]:
            self._ensure_unique_emp_code(str(fields["emp_code"]), exclude_id=emp_id)

        # Serialize date fields
        if "date_joined" in fields:
            fields["date_joined"] = _iso(fields["date_joined"])

        q = Query()
        return bool(self.table.update(fields, q.id == emp_id))

    def delete_employee(self, emp_id: int) -> bool:
        q = Query()
        return bool(self.table.remove(q.id == emp_id))

    # ---------- queries ----------
    def get_by_id(self, emp_id: int) -> Optional[Dict[str, Any]]:
        q = Query()
        res = self.table.get(q.id == emp_id)
        return dict(res) if res else None

    def find_by_emp_code(self, emp_code: str) -> Optional[Dict[str, Any]]:
        q = Query()
        res = self.table.get(q.emp_code == emp_code)
        return dict(res) if res else None

    def list_all(self) -> List[Dict[str, Any]]:
        return [dict(r) for r in self.table.all()]

    def active_employees(self) -> List[Dict[str, Any]]:
        q = Query()
        return [dict(r) for r in self.table.search(q.is_active == True)]
